//package globalExamples;

import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.StringSource;

import oracle.olapi.transaction.Transaction;
import oracle.olapi.transaction.TransactionProvider;
import oracle.olapi.transaction.NotCommittableException;

import oracle.olapi.metadata.mdm.MdmDimensionMemberInfo;
import oracle.olapi.metadata.mdm.MdmHierarchy;
import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;

/**
 * Complete code for the examples in Chapter 8, Using a TransactionProvider,
 * in the Oracle OLAP Developer's Guide to the  OLAP API.
 *
 * This program uses the Context10g class, which uses the
 * CursorPrintWriter class. It also uses the TopBottomTemplate and
 * SingleSelectionTemplate classes that are described in Chapter 11,
 * Creating Dynamic Queries.
 * It uses much of the same code as the TopBottomTest class, which is
 * Example 11-4 in Chapter 11.
 *
 * @author Oracle Corporation
 */
public class TransactionExamples
{
  public TransactionExamples()
  {
  }

   public void run(String[] args)
  {
    Context10g context = new Context10g(args, false);
    DataProvider dp = context.getDataProvider();
    TransactionProvider tp = context.getTransactionProvider();

    // Create a SingleSelectionTemplate.
    // Get the MdmMeasure for the measure.
    MdmMeasure mdmUnits = context.getMdmMeasureByName("UNITS");

    // Get the Source for the measure.
    Source units = mdmUnits.getSource();

    // Get the MdmPrimaryDimension objects for the dimensions of the measure.
    MdmPrimaryDimension[] mdmPrimDims =
           context.getMdmPrimaryDimensionsByName(new String[] {"CUSTOMER",
                                                               "PRODUCT",
                                                               "CHANNEL",
                                                               "TIME"});
    MdmPrimaryDimension mdmCustDim = mdmPrimDims[0];
    MdmPrimaryDimension mdmProdDim = mdmPrimDims[1];
    MdmPrimaryDimension mdmChanDim = mdmPrimDims[2];
    MdmPrimaryDimension mdmTimeDim = mdmPrimDims[3];

    // Get the default hierarchy of the Product dimension.
    MdmHierarchy mdmProdRollup = mdmProdDim.getDefaultHierarchy();

    // Get the StringSource for the hierarchy.
    StringSource prodRollup = (StringSource) mdmProdRollup.getSource();

    // Create a SingleSelectionTemplate to produce a Source that
    // represents the measure values specified by single members of each of
    // the dimensions of the measure other than the base dimension.
    SingleSelectionTemplate singleSelections =
                        new SingleSelectionTemplate(units, dp);

    // Create MdmDimensionMemberInfo objects for single members of the
    // other dimensions of the measure.
    MdmDimensionMemberInfo timeMemInfo =
         new MdmDimensionMemberInfo(mdmTimeDim, "CALENDAR::YEAR::4");
    MdmDimensionMemberInfo custMemInfo =
         new MdmDimensionMemberInfo(mdmCustDim, "SHIPMENTS_ROLLUP::REGION::8");
    MdmDimensionMemberInfo chanMemInfo =
         new MdmDimensionMemberInfo(mdmChanDim, "CHANNEL_ROLLUP::CHANNEL::2");

    // Add the dimension member information objects to the
    // SingleSelectionTemplate.
    singleSelections.addDimMemberInfo(custMemInfo);
    singleSelections.addDimMemberInfo(chanMemInfo);
    singleSelections.addDimMemberInfo(timeMemInfo);

    context.println("\nExample 8-1, Rolling Back a Transaction");

    // ***** Beginning of the Rolling Back a Transaction example *****
    // The current Transaction is a read Transaction, t1.
    // Create a TopBottomTemplate using a hierarchy of the Product dimension
    // as the base and dp as the DataProvider.
    TopBottomTemplate topNBottom = new TopBottomTemplate(prodRollup, dp);

    // Changing the state of a Template requires a write Transaction, so a
    // write child Transaction, t2, is automatically started.
    topNBottom.setTopBottomType(TopBottomTemplate.TOP_BOTTOM_TYPE_TOP);
    topNBottom.setN(10);
    topNBottom.setCriterion(singleSelections.getSource());

    // Prepare and commit the Transaction t2.
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      context.println("Cannot commit the Transaction. " + e);
    }
    tp.commitCurrentTransaction();            //t2 disappears

    // The current Transaction is now t1.
    // Get the dynamic Source produced by the TopBottomTemplate.
    Source result = topNBottom.getSource();

    // Create a Cursor and display the results (these operations are
    // performed by the context object).
    context.println("\nThe current state of the TopBottomTemplate\n" +
                    "produces the following values:\n");
    context.displayTopBottomResult(result);

    // Start a child Transaction, t3. It is a read Transaction.
    tp.beginSubtransaction();         // t3 is the current Transaction

    // Change the state of topNBottom. Changing the state requires a
    // write Transaction so Transaction t4 starts automatically,
    topNBottom.setTopBottomType(TopBottomTemplate.TOP_BOTTOM_TYPE_BOTTOM);
    topNBottom.setN(15);

    // Prepare and commit the Transaction.
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      context.println("Cannot commit the Transaction. " + e);
    }
    tp.commitCurrentTransaction();           // t4 disappears

    // Create a Cursor and display the results. // t3 is the current Transaction
    context.println("\nIn the child Transaction, the state of the\n" +
                    "TopBottomTemplate produces the following values:\n");
    context.displayTopBottomResult(result);

    // The displayTopBottomResult method closes the CursorManager for the
    // Cursor created in t3.

    // Undo t3, which discards the state of topNBottom that was set in t4.
    tp.rollbackCurrentTransaction();         // t3 disappears

    // Transaction t1 is now the current Transaction and the state of
    // topNBottom is the one defined in t2.

    // To show the current state of the TopNBottom template Source,
    // prepare and commit the Transaction, create a Cursor, and display
    // its values.
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      context.println("Cannot commit the Transaction " + e);
    }
    tp.commitCurrentTransaction();

    context.println("\nAfter rolling back the child Transaction, the state "
                    + "of\nthe TopBottomTemplate produces the following "
                    + "values:\n");
    context.displayTopBottomResult(result);

    // ***** End of the Rolling Back a Transaction example *****


    context.println("\nExample 8-2, Using Child Transaction Objects");

    // ***** Beginning of the Using Child Transaction Objects example *****

    // The parent Transaction is the current Transaction at this point.
    // Save the parent read Transaction as parentT1.
    Transaction parentT1 = tp.getCurrentTransaction();

    // Get the dynamic Source produced by the TopBottomTemplate.
    // The next line is unnecessary because result already exists.
    //Source result = topNBottom.getSource();

    // Create a Cursor and display the results (these operations are
    // performed by the context object).
    context.println("\nThe current state of the TopBottomTemplate\n" +
                    "produces the following values:\n");
    context.displayTopBottomResult(result);

    // Begin a child Transaction of parentT1.
    tp.beginSubtransaction();  // This is a read Transaction.

    // Save the child read Transaction as childT2.
    Transaction childT2 = tp.getCurrentTransaction();

    // Change the state of the TopBottomTemplate. This starts a
    // write Transaction, a child of the read Transaction childT2.
    topNBottom.setN(12);
    topNBottom.setTopBottomType(TopBottomTemplate.TOP_BOTTOM_TYPE_BOTTOM);

    // Save the child write Transaction as writeT3.
    Transaction writeT3 = tp.getCurrentTransaction();

    // Prepare and commit the write Transaction writeT3.
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
     context.println("Cannot commit current Transaction. " + e);
    }
    tp.commitCurrentTransaction();

    // The commit moves the changes made in writeT3 into its parent,
    // the read Transaction childT2. The writeT3 Transaction
    // disappears. The current Transaction is now childT2
    // again but the state of the TopBottomTemplate has changed.

    // Create a Cursor and display the results of the changes to the
    // TopBottomTemplate that are visible in childT2.
    try
    {
      context.println("\nIn the child Transaction, the state of the\n" +
                      "TopBottomTemplate produces the following values:\n");

      context.displayTopBottomResult(result);
    }
    catch(Exception e)
    {
      context.println("Cannot display the results of the query. " + e);
    }

    // Begin a grandchild Transaction of the initial parent.
    tp.beginSubtransaction();  // This is a read Transaction.

    // Save the grandchild read Transaction as grandchildT4.
    Transaction grandchildT4 = tp.getCurrentTransaction();

    // Change the state of the TopBottomTemplate. This starts another
    // write Transaction, a child of grandchildT4.
    topNBottom.setTopBottomType(TopBottomTemplate.TOP_BOTTOM_TYPE_TOP);

    // Save the write Transaction as writeT5.
    Transaction writeT5 = tp.getCurrentTransaction();

    // Prepare and commit writeT5.
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      context.println("Cannot commit current Transaction. " + e);
    }
    tp.commitCurrentTransaction();

    // Transaction grandchildT4 is now the current Transaction and the
    // changes made to the TopBottomTemplate state are visible.

    // Create a Cursor and display the results visible in grandchildT4.
    try
    {
      context.println("\nIn the grandchild Transaction, the state of the\n" +
                      "TopBottomTemplate produces the following values:\n");

      context.displayTopBottomResult(result);
    }
    catch(Exception e)
    {
      context.println("Cannot display the results of the query. " + e);
    }

    // Commit the grandchild into the child.
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      context.println("Cannot commit current Transaction. " + e);
    }
    tp.commitCurrentTransaction();

    // Transaction childT2 is now the current Transaction.
    // Instead of preparing and committing the grandchild Transaction,
    // you could rollback the Transaction, as in the following
    // method call:
    //   tp.rollbackCurrentTransaction();
    // If you roll back the grandchild Transaction, then the changes
    // you made to the TopBottomTemplate state in the grandchild
    // are discarded and childT2 is the current Transaction.

    // Commit the child into the parent.
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException nc)
    {
      context.println("Cannot prepare the child Transaction for committing. "
                      + nc);
    }
    catch(Exception e)
    {
      context.println("\nCannot prepare the child Transaction for committing."
                      + " Encountered exception \n" + e + ".");
    }
    try
    {
      tp.commitCurrentTransaction();

      // Create a Cursor and display the results after committing the
      // grandchild and the child.
      context.println("\nAfter committing the grandchild and the child, " +
                      "the state of the\n TopBottomTemplate produces the " +
                      "following values:\n");

      context.displayTopBottomResult(result);
    }
    catch(Exception e)
    {
      context.println("\nCannot commit the child Transaction. Encountered " +
                       "exception \n" + e + ".");
    }

    // Transaction parentT1 is now the current Transaction. Again,
    // you can roll back the childT2 Transaction instead of
    // preparing and committing it. If you do so, then the changes
    // you made in childT2 are discarded. The current Transaction
    // is parentT1, which has the original state of the TopBottomTemplate,
    // without any of the changes made in the grandchild or the
    // child transactions.

    // ***** End of the Using Child Transaction Objects example *****

  }

  public static void main(String[] args)
  {
    new TransactionExamples().run(args);
  }
}
